﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using MicroRWD.Common;

namespace MicroRWD.MFIC
{
    // Quad tag reader
    public class MFICReader : Reader
    {
        #region Private Constants

        // The read block command (ASCII 'R')
        private byte C_CMD_BLOCK_READ = 0x52;

        // The write block command (ASCII 'W')
        private byte C_CMD_BLOCK_WRITE = 0x57;

        // The uid command (ASCII 'U')
        private byte C_CMD_UID = 0x55;

        // The store keys command (ASCII 'K')
        private byte C_CMD_KEY_STORE = 0x4B;

        #endregion

        #region Private Attributes

        // Reader status changed event handler
        private event StatusUpdateEventHandler statusUpdateEvent;

        #endregion

        #region Public Properties

        // ICODE Card UID
        private byte[] lastIcodeUid = null;
        public string LastIcodeUidStr
        {
            get
            {
                // Build the UID String byte by byte
                string uidString = "";
                if ((lastIcodeUid != null) && (lastIcodeUid.Length == 8))
                {
                    for (int i = 0; i < lastIcodeUid.Length; i++)
                    {
                        if (i == (lastIcodeUid.Length - 1))
                        {
                            // No space after the last digit
                            uidString += lastIcodeUid[i].ToString("X2");
                        }
                        else
                        {
                            uidString += lastIcodeUid[i].ToString("X2") + " ";
                        }
                    }
                }
                return uidString;
            }
        }

        // Mifare Card UID
        private byte[] lastMifareUid = null;
        public string LastMifareUidStr
        {
            get
            {
                // Build the UID String byte by byte
                string uidString = "";
                if ((lastMifareUid != null) && (lastMifareUid.Length == 7))
                {
                    for (int i = 0; i < lastMifareUid.Length; i++)
                    {
                        if (i == (lastMifareUid.Length - 1))
                        {
                            // No space after the last digit
                            uidString += lastMifareUid[i].ToString("X2");
                        }
                        else
                        {
                            uidString += lastMifareUid[i].ToString("X2") + " ";
                        }
                    }
                }
                return uidString;
            }
        }

        // Mode
        public Mode Mode { get; private set; }

        // Status update event handler
        public event StatusUpdateEventHandler StatusUpdateEvent { add { statusUpdateEvent += value; } remove { statusUpdateEvent -= value; } }

        #endregion

        #region Constructor

        // Constructs a new MFIC reader instance attached via the specified port name
        public MFICReader(string _portName, int _commandTimeoutMillis) : base(_portName, _commandTimeoutMillis)
        {
            // Initialise mode to none
            Mode = Mode.NONE;

            PollingFunction = CmdUIDRead;

            // Register for acknowledge events
            base.AcknowledgeEvent += MyAcknowledgeEventHandler;
        }

        #endregion
        
        #region Public Methods

        // Card/Label Status (Mifare/ICODE)
        public new Status CmdStatus()
        {
            Status result = new Status();

            // Delegate to generic reader
            byte ack = base.CmdStatus();

            // Check for success (got response)
            if ((ack & 0x80) == 0x80)
            {
                result = new Status(ack);
            }

            return result;
        }

        // Program EEPROM (Mifare/ICODE)
        public new bool CmdProgram(byte location, byte value)
        {
            bool result = false;

            // Delegate to generic reader
            byte ack = base.CmdProgram(location, value);

            // Check for success (got response with clear status flags)
            if ((ack & 0x89) == 0x80)
            {
                result = true;
            }

            return result;
        }

        // Select reader type (Mifare/ICODE)
        public bool CmdReaderType(ReaderType _type)
        {
            bool result = false;

            // Set Mifare/ICODE option byte in RWD EEPROM
            result = CmdProgram(0x03, (byte)_type);  

            // Return result
            return result;
        }
        
        // Read Card/Label UID (Mifare/ICODE)
        public byte[] CmdUIDRead()
        {
            // Delegate to base class
            byte[] result = CmdGeneric(new byte[] { C_CMD_UID });

            // Clear the last UID
            lastIcodeUid = null;
            lastMifareUid = null;

            // Check the status byte for success
            if ((result != null) && (result.Length > 0))
            {
                if ((result[0] & 0x86) == 0x86)
                {
                    if (result.Length == (1 + 8))
                    {
                        // Save as an ICODE UID
                        if ((lastIcodeUid == null)
                            || (lastIcodeUid.Length != 8))
                        {
                            lastIcodeUid = new byte[8];
                        }
                        Array.Copy(result, 1, lastIcodeUid, 0, lastIcodeUid.Length);
                    }
                    else if (result.Length == (1 + 7))
                    {
                        // Save as a Mifare UID
                        if ((lastMifareUid == null)
                            || (lastMifareUid.Length != 7))
                        {
                            lastMifareUid = new byte[7];
                        }
                        Array.Copy(result, 1, lastMifareUid, 0, lastMifareUid.Length);
                    }
                }
            }
            return result;
        }

        // Read Card Block (Mifare)
        public byte[] CmdCardBlockRead(byte _blockAddr, bool _keyType, byte _keyCode)
        {
            // Zero length reply on timeout, data otherwise
            byte[] result = new byte[0];

            // Check mode and parameters
            if ((Mode == Mode.MIFARE)
                && (_blockAddr >= 0) && (_blockAddr <= 0xff) 
                && (_keyCode >= 0) && (_keyCode <= 0x1f))
            {
                // Create the key parameter from the key type and key code
                byte keyArgument;
                keyArgument = (byte)(_keyType ? 1 << 7 : 0);
                keyArgument |= _keyCode;

                // Delegate to base class
                result = CmdGeneric(new byte[] { C_CMD_BLOCK_READ, _blockAddr, keyArgument });
            }
            else
            {
                // Error
                Log.Error("invalid parameters");
            }

            return result;
        }

        // Write Card Block (Mifare)
        public byte[] CmdCardBlockWrite(byte _blockAddr, bool _keyType, byte _keyCode, byte[] _data)
        {
            byte[] result = new byte[0];

            // Check mode and parameters
            if ((Mode == Mode.MIFARE) 
                && (_blockAddr >= 0) && (_blockAddr <= 0xff) 
                && (_keyCode >= 0) && (_keyCode <= 0x1f) 
                && (_data.Length == 16))
            {
                // Build the message
                byte[] command = new byte[19];
                command[0] = C_CMD_BLOCK_WRITE;
                command[1] = _blockAddr;                
                command[2] = (byte)(_keyType ? 1 << 7 : 0);
                command[2] |= _keyCode;
                Array.Copy(_data, 0, command, 3, _data.Length);

                // Delegate to base class
                result = CmdGeneric(command);
            }
            else
            {
                // Error
                Log.Error("invalid parameters");
            }

            return result;
        }

        // Store Key (Mifare)
        public byte[] CmdKeyStore(byte _keyCode, byte[] _data)
        {
            byte[] result = new byte[0];

            // Check mode and parameters
            if ((Mode == Mode.MIFARE)
                && (_keyCode >= 0) && (_keyCode <= 0x1f) 
                && (_data.Length == 6))
            {
                // Build the message
                byte[] command = new byte[8];
                command[0] = C_CMD_KEY_STORE;
                command[1] = _keyCode;
                Array.Copy(_data, 0, command, 2, _data.Length);

                // Delegate to base class
                result = CmdGeneric(command);
            }
            else
            {
                // Error
                Log.Error("invalid parameters");
            }

            return result;
        }


        // Read Label Block (ICODE)
        public byte[] CmdLabelBlockRead(byte _blockAddr)
        {
            // Zero length reply on timeout, data otherwise
            byte[] result = new byte[0];

            if (lastIcodeUid != null)
            {
                // Check mode and parameters
                if ((Mode == Mode.ICODE)
                    && (_blockAddr >= 0) && (_blockAddr <= 0x1B))
                {
                    // Build the message
                    byte[] command = new byte[10];
                    command[0] = C_CMD_BLOCK_READ;
                    command[1] = _blockAddr;
                    Array.Copy(lastIcodeUid, 0, command, 2, lastIcodeUid.Length);

                    // Delegate to base class
                    result = CmdGeneric(command);
                }
                else
                {
                    // Error
                    Log.Error("invalid parameters");
                }
            }
            return result;
        }

        // Write Label Block (ICODE)
        public byte[] CmdLabelBlockWrite(byte _blockAddr, byte[] _data)
        {
            // Zero length reply on timeout, data otherwise
            byte[] result = new byte[0];

            if (lastIcodeUid != null)
            {
                // Check mode and parameters
                if ((Mode == Mode.ICODE)
                    && (_blockAddr >= 0) && (_blockAddr <= 0x1B))
                {
                    // Build the message
                    byte[] command = new byte[14];
                    command[0] = C_CMD_BLOCK_WRITE;
                    command[1] = _blockAddr;
                    Array.Copy(lastIcodeUid, 0, command, 2, lastIcodeUid.Length);
                    Array.Copy(_data, 0, command, 10, _data.Length);

                    // Delegate to base class
                    result = CmdGeneric(command);
                }
                else
                {
                    // Error
                    Log.Error("invalid parameters");
                }
            }
            return result;
        }

        // Set mode
        public bool SetMode(Mode _mode)
        {
            bool result = false;

            switch (_mode)
            {
                case Mode.MIFARE:
                    if (CmdReaderType(ReaderType.Mifare))
                    {
                        Mode = Mode.MIFARE;
                        result = true;
                    }
                    break;

                case Mode.ICODE:
                    if (CmdReaderType(ReaderType.Icode))
                    {
                        Mode = Mode.ICODE;
                        result = true;
                    }
                    break;
            }

            return result;
        }

        #endregion
        
        #region Private Methods

        // Fire status update event to listeners
        private void FireStatusUpdateEvent(Status _status)
        {
            try
            {
                // Instantiate suitable event args object
                StatusUpdateEventArgs args = new StatusUpdateEventArgs(_status);

                // Check for current listeners
                if (statusUpdateEvent != null)
                {
                    // Dispatch event to listeners
                    statusUpdateEvent(this, args);
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex.ToString());
            }
        }

        // Handle acknowledge events from base reader
        private void MyAcknowledgeEventHandler(object sender, AcknowledgeEventArgs args)
        {
            // Decode byte into Status instance
            Status status = new Status(args.Value);

            // Notify listeners using a Status instance
            FireStatusUpdateEvent(status);
        }

        #endregion
    }
}
